import sys
import logging

from owslib.util import nspath_eval
from owslib.wps import WebProcessingService, FeatureCollection, namespaces
from owslib.etree import etree

def main():
    inputFile = sys.argv[1]
    outputFile = sys.argv[2]

    wps = WebProcessingService('https://rain1.fsv.cvut.cz/services/wps', skip_caps=True)
    processId = 'd-rain6h-timedist'

    inputs = [
        ("input", GML(createPolygons(inputFile))),
        ("keycolumn", "cat"),
        ("return_period", "N10"),
        ("type", "A,B,C,D,E,F"),
        ("area_red", "false"),
    ]

    execution = wps.execute(processId, inputs)
    execution.getOutput(outputFile)

def createPolygons(inputFile):
    polygons = [[]]
    with open(inputFile) as f:
        for line in f:
            if line.strip():
                polygons[-1].append(tuple(map(float, line.split())))
            else:
                polygons.append([])

    polygons = [pol for pol in polygons if len(pol) > 2]

    for pol in polygons:
        if pol[-1] != pol[0]:
            pol.append(pol[0])

    polygons = [switchAllCoords(pol) for pol in polygons]

    return polygons


def switchAllCoords(pol):
    return [switchCoords(p) for p in pol]


def switchCoords(point):
    x, y = point[:]
    return (-y, -x)

class GML(FeatureCollection):
    def __init__(self, polygons):
        super().__init__()
        self.polygons = polygons

    def getXml(self):
        ogrspace = { 'ogr' : "http://ogr.maptools.org/" }
        dataElement = etree.Element(nspath_eval('wps:Data', namespaces))
        complexDataElement = etree.SubElement(dataElement, nspath_eval('wps:ComplexData', namespaces), attrib={"mimeType": "text/xml"})
        featureCollection = etree.SubElement(complexDataElement, nspath_eval('ogr:FeatureCollection', ogrspace))
        for polygon in self.polygons:
            featureMember = etree.SubElement(featureCollection, nspath_eval('gml:featureMember', namespaces))
            test = etree.SubElement(featureMember, nspath_eval('ogr:test', ogrspace), attrib={'fid': "test.0"})
            geometryProperty = etree.SubElement(test, nspath_eval('ogr:geometryProperty', ogrspace))
            polygo = etree.SubElement(geometryProperty, nspath_eval('gml:Polygon', namespaces), attrib={'srsName': "EPSG:5514"})
            outerBoundaryIs = etree.SubElement(polygo, nspath_eval('gml:outerBoundaryIs', namespaces))
            linearRing = etree.SubElement(outerBoundaryIs, nspath_eval('gml:LinearRing', namespaces))
            coordinates = etree.SubElement(linearRing, nspath_eval('gml:coordinates', namespaces))
            coordinates.text = ' '.join(["%s,%s" % (x, y) for x, y in polygon[:]])
        return dataElement

log = logging.getLogger('owslib')
log.addHandler(logging.StreamHandler())
log.setLevel(logging.DEBUG)

if __name__ == "__main__":
    main()
